/*-----------------------------------------------------------------------------
 * Enhydra Java Application Server
 * Copyright 1997-2000 Lutris Technologies, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes Enhydra software developed by Lutris
 *      Technologies, Inc. and its contributors.
 * 4. Neither the name of Lutris Technologies nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY LUTRIS TECHNOLOGIES AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL LUTRIS TECHNOLOGIES OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *-----------------------------------------------------------------------------
 * ../../kb/data/InfoQuery.java
 *-----------------------------------------------------------------------------
 */

package kb.data;

import com.lutris.dods.builder.generator.query.*;
import com.lutris.dods.builder.generator.dataobject.GenericDO;

import com.lutris.appserver.server.*;
import com.lutris.appserver.server.sql.*;
import com.lutris.appserver.server.session.*;

import java.sql.*;
import java.util.*;
import java.math.*;

import java.util.Date;  // when I say Date, I don't mean java.sql.Date

/**
 * InfoQuery is used to query the info table in the database.<BR>
 * It returns objects of type InfoDO.
 * <P>
 * General usage:
 * <P>
 *     In DODS:
 *        Create a Data Object named "Dog",
 *        and create an Attribute named "Name",
 *        and set that Attribute to "Can be queried."
 *        DODS will then generate the method DogQuery.setQueryName().
 * <P>
 *     In your Business Layer, prepare the query:<BR>
 * <P><PRE>
 *             DogQuery dq = new DogQuery();
 *             dq.setQueryName("Rex")
 *             if ( Rex is a reserved name )
 *                 dq.requireUniqueInstance();
 * </PRE>
 *     Then, get the query results one of two ways:
 * <P>
 *         #1:<PRE>
 *             String names = "";
 *             DogDO[] dogs = dq.getDOArray();
 *             for ( int i = 0; i < dogs.length; i++ ) {
 *                 names += dogs[i].getName() + " ";
 *             }
 * </PRE>
 *      or #2:<PRE>
 *             String names = "";
 *             DogDO dog;
 *             while ( null != ( dog = dq.getNextDO() ) ) {
 *                 names += dog.getName() + " ";
 *             }
 * </PRE>
 * <P>
 *     Note:   If <CODE>requireUniqueInstance()</CODE> was called,
 *             then <CODE>getDOArray()</CODE> or <CODE>getNextDO()</CODE>
 *             will throw an exception if more than one "Rex" was found.
 * <P>
 *     Note:   Results of the query will come from the Data Object cache if:
 *             -  The cache is available.
 *             -  Matches were found in the cache.
 *             -  No other tables (Data Objects of other types) were involved
 *                in the query.
 *                This can happen if you extend the <CODE>DogQuery</CODE> class
 *                and you make calls to the <CODE>QueryBuilder</CODE> object
 *                to add SQL involving other tables.
 *             If any of these conditions is not true,
 *             then any results from the query will come from the database.
 * <P>
 *     To reuse the query object, call:
 * <P><PRE>
 *             dq.reset();
 * </PRE>
 * @author Administrateur
 * @version $Revision: 1.5.2.2.2.1 $
 */
public class InfoQuery implements Query {

    private QueryBuilder builder;

    /**
     * Public constructor.
     */
    public InfoQuery() {
	builder = new QueryBuilder( "info", "info.*" );
	builder.setDatabaseVendor( "Standard" );
	builder.setStringMatchDetails( "LIKE", "%" );
	reset();
    }

    private ResultSet		resultSet	= null;
    private boolean 		uniqueInstance	= false;
    private InfoDO[]	DOs		= null;
    private int			arrayIndex	= -1;
    private boolean		needToRun	= true;
    private Vector		cacheHits	= null;

    private boolean partialCache = false;
    private boolean hitDb = true;
    public void hitDatabase() { hitDb = true; }

    /**
     * Perform the query on the database, and prepare the
     * array of returned DO objects.
     *
     * @exception DataObjectException If a database access error occurs.
     * @exception NonUniqueQueryException If too many rows were found.
     */
    private void runQuery()
    throws DataObjectException, NonUniqueQueryException {
	needToRun = false;
	arrayIndex = -1;

	DBQuery dbQuery = null;
	try {
	    Vector results;
	    if ( partialCache && 0 == cacheHits.size() )
		hitDb = true;
	    if ( hitDb ) {
		dbQuery = Enhydra.getDatabaseManager().createQuery();
		dbQuery.query( this );    // invokes executeQuery
	        results = new Vector();
	        while ( resultSet.next() ) {
		    results.addElement( InfoDO.createExisting ( resultSet ) );
	        }
	    } else {
		results = cacheHits;	 // executeQuery saw cache hits
	    }

	    if ( results.size() > 1 && uniqueInstance )
		throw new NonUniqueQueryException(
		    "Too many rows returned from database" );
	    DOs = new InfoDO [ results.size() ];
	    for ( int i = 0; i < results.size(); i++ ) {
		DOs[i] = ( InfoDO ) results.elementAt( i );
	    }
	    arrayIndex = 0;
	} catch ( SQLException se ) {
	    if (null == se.getSQLState() ) {
		throw new DataObjectException(
		    "Unknown SQLException", se );
	    }
	    if (	se.getSQLState().startsWith("02") &&
			se.getErrorCode() == 100 ) {
		throw new DataObjectException(
		    "Update or delete DO is out of synch", se );
	    } else if (	se.getSQLState().equals("S1000") &&
			se.getErrorCode() == -268 ) {
		throw new DataObjectException(
		    "Integrity constraint violation", se );
	    } else {
		throw new DataObjectException( "Data Object Error", se );
	    }
	} catch ( ObjectIdException oe ) {
	    throw new DataObjectException( "Object ID Error", oe );
	} catch ( DatabaseManagerException de ) {
	    throw new DataObjectException( "Database connection Error", de );
	} finally {
	    if ( null != dbQuery )
		dbQuery.release();
	}
    }


    /**
     * Limit the number of rows (DOs) returned.
     * NOTE: When setting a limit on rows returned by a query,
     * you usually want to use a call to an addOrderBy method
     * to cause the most interesting rows to be returned first.
     * @exception DataObjectException If a database access error occurs.
     * @exception NonUniqueQueryException If too many rows were found.
     */
    public void setMaxRows( int maxRows )
    throws DataObjectException, NonUniqueQueryException {
	hitDb = true;
	builder.setMaxRows( maxRows );
    }

    /**
     * Return array of DOs constructed from ResultSet returned by query.
     * @exception DataObjectException If a database access error occurs.
     * @exception NonUniqueQueryException If too many rows were found.
     */
    public InfoDO[] getDOArray()
    throws DataObjectException, NonUniqueQueryException {
	if ( needToRun )
	    runQuery();
	return DOs;
    }

    /**
     * Return successive DOs from array built from ResultSet returned by query.
     * @exception DataObjectException If a database access error occurs.
     * @exception NonUniqueQueryException If too many rows were found.
     */
    public InfoDO getNextDO()
    throws DataObjectException, NonUniqueQueryException {
	if ( needToRun )
	    runQuery();
	if ( null == DOs ) {
	    /* This should never happen.
	     * runQuery() should either throw an exception
	     * or create an array of DOs, possibly of zero length.
	     */
	    return null;
	}
	if ( arrayIndex < DOs.length )
	    return DOs[ arrayIndex++ ];
	return null;
    }

    /**
     * Set the OID to query.
     * WARNING!  This method assumes that table <CODE>info</CODE>
     * has a column named <CODE>"oid"</CODE>.
     * This method is called from the DO classes to retrieve an object by id.
     *
     * @param oid The object id to query.
     */
    public void setQueryOId(ObjectId oid) {
        // Remove from cacheHits any DOs that do not meet this
        // setQuery requirement.
	if ( null == oid )
	    return;
	requireUniqueInstance();
        for ( int i = 0; i < cacheHits.size(); i++ ) {
            InfoDO DO = ( InfoDO ) cacheHits.elementAt( i );
            if ( null == DO ) continue;
            boolean equals = true;
	    ObjectId id = DO.getOId();
	    if ( null == id || ! id.equals( oid ) )
		cacheHits.removeElementAt( i-- );
	}

	// Also prepare the SQL needed to query the database
        // in case there is no cache, or the query involves other tables.
	builder.addWhere( InfoDO.PrimaryKey, oid.toBigDecimal(),
				QueryBuilder.EQUAL );
    }

    /**
     * Set the object handle to query.
     * This is a variant of setQueryOId().
     *
     * @param handle The string version of the id to query.
     */
    public void setQueryHandle(String handle)
    throws ObjectIdException {
        ObjectId oid = new ObjectId( handle );
        setQueryOId( oid );
    }

    /**
     * Set "unique instance" assertion bit.
     * The first call to the next() method will throw an exception
     * if more than one object was found.
     */
    public void requireUniqueInstance()
    {
	uniqueInstance = true;
    }

    /**
     * Reset the query parameters.
     */
    public void reset() {
	cacheHits	= new Vector();
	DOs		= null;
	uniqueInstance	= false;
	needToRun	= true;
        builder.reset();
    }

    /**
     * Return the appropriate QueryBuilder flag for selecting
     * exact matches (SQL '=') or inexact matches (SQL 'matches').
     */
    private boolean exactFlag( boolean exact ) {
        return exact ? QueryBuilder.EXACT_MATCH : QueryBuilder.NOT_EXACT;
    }



    //
    // Implementation of Query interface
    //

    /**
     * Method to query objects from the database.
     * The following call in runQuery()
     *	    dbQuery.query( this );
     * causes the dbQuery object to invoke
     *      executeQuery()
     *
     * @param conn Handle to database connection.
     * @exception java.sql.SQLException If a database access error occurs.
     */
    public ResultSet executeQuery(DBConnection conn)
        throws SQLException
    {
	resultSet = builder.executeQuery( conn );
        return resultSet;
    }


    /**
     * WARNING!  This method is disabled.
     * It's implementation is forced by the Query interface.
     * This method is disabled for 2 reasons:
     * 1)  the getDOArray() and getNextDO() methods are better
     *     because they return DOs instead of JDBC objects.
     * 2)  the createExisting() method throws an exception
     *     that we cannot reasonably handle here,
     *     and that we cannot throw from here.
     *
     * @param rs JDBC result set from which the next object
     *   will be instantiated.
     * @exception java.sql.SQLException
     *   If a database access error occurs.
     * @exception com.lutris.appserver.server.sql.ObjectIdException
     *   If an invalid object id was queried from the database.
     */
    public Object next(ResultSet rs)
    throws SQLException, ObjectIdException {
	// TODO: It would be nice to throw an unchecked exception here
	// (an exception that extends RuntimeException)
	// that would be guaranteed to appear during application testing.
	throw new ObjectIdException(
	    "next() should not be used.  Use getNextDO() instead." );
	//return null;
    }


    /**
     * Set the type to query, with a QueryBuilder comparison operator.
     *
     * @param x The type of the info to query.
     * @param cmp_op QueryBuilder comparison operator to use.
     * @exception DataObjectException If a database access error occurs.
     * @exception QueryException If comparison operator is inappropriate
     * (e.g. CASE_SENSITIVE_CONTAINS on an integer field).
     */
    public void setQueryType(
				String x,
				String cmp_op )
    throws DataObjectException, QueryException
    {
	// Remove from cacheHits any DOs that do not meet this
	// setQuery requirement.
	for ( int i = 0; i < cacheHits.size() && ! hitDb; i++ ) {
	    InfoDO DO = ( InfoDO ) cacheHits.elementAt( i );
	    if ( null == DO ) continue;
	    boolean found = false;
	    String m = DO.getType();
	    found = QueryBuilder.compare( m, x, cmp_op );
	    if ( ! found )
		cacheHits.removeElementAt( i-- );
	}

	// Also prepare the SQL needed to query the database 
	// in case there is no cache, or the query involves other tables.
	if ( partialCache || hitDb )
	    builder.addWhere( InfoDO.Type, x, cmp_op );
    }

    /**
     * Set the type to query.
     *
     * @param x The type of the info to query.
     * @param exact to use matches or not
     * @exception DataObjectException If a database access error occurs.
     * @deprecated Instead use 
     * setQueryType(String x,String cmp_op)
     */
    public void setQueryType(
				String x, boolean exact)
    throws DataObjectException, QueryException
    {
	// TODO:  DELETE THIS METHOD IN A FUTURE RELEASE
	String cmp_op = QueryBuilder.EQUAL;
	if ( ! exact )
	    cmp_op = QueryBuilder.CASE_INSENSITIVE_CONTAINS;
	setQueryType( x, cmp_op );
    }

    /**
     * Set the type to query
     * @param x The type of the info to query.
     * @exception DataObjectException If a database access error occurs.
     */
    public void setQueryType( 
				String x )
    throws DataObjectException, QueryException {
	setQueryType( x, true );
    }

    /**
     * Add type to the ORDER BY clause.
     *
     * @param direction_flag  True for ascending order, false for descending
     */
    public void addOrderByType(boolean direction_flag) {
	hitDb = true;
        builder.addOrderByColumn("type",
					(direction_flag) ? "ASC" : "DESC");
    }


    /**
     * Add type to the ORDER BY clause.  This convenience
     * method assumes ascending order.
     */
    public void addOrderByType() {
        builder.addOrderByColumn("type","ASC");
    }


    /**
     * Set the titre to query, with a QueryBuilder comparison operator.
     *
     * @param x The titre of the info to query.
     * @param cmp_op QueryBuilder comparison operator to use.
     * @exception DataObjectException If a database access error occurs.
     * @exception QueryException If comparison operator is inappropriate
     * (e.g. CASE_SENSITIVE_CONTAINS on an integer field).
     */
    public void setQueryTitre(
				String x,
				String cmp_op )
    throws DataObjectException, QueryException
    {
	// Remove from cacheHits any DOs that do not meet this
	// setQuery requirement.
	for ( int i = 0; i < cacheHits.size() && ! hitDb; i++ ) {
	    InfoDO DO = ( InfoDO ) cacheHits.elementAt( i );
	    if ( null == DO ) continue;
	    boolean found = false;
	    String m = DO.getTitre();
	    found = QueryBuilder.compare( m, x, cmp_op );
	    if ( ! found )
		cacheHits.removeElementAt( i-- );
	}

	// Also prepare the SQL needed to query the database 
	// in case there is no cache, or the query involves other tables.
	if ( partialCache || hitDb )
	    builder.addWhere( InfoDO.Titre, x, cmp_op );
    }

    /**
     * Set the titre to query.
     *
     * @param x The titre of the info to query.
     * @param exact to use matches or not
     * @exception DataObjectException If a database access error occurs.
     * @deprecated Instead use 
     * setQueryTitre(String x,String cmp_op)
     */
    public void setQueryTitre(
				String x, boolean exact)
    throws DataObjectException, QueryException
    {
	// TODO:  DELETE THIS METHOD IN A FUTURE RELEASE
	String cmp_op = QueryBuilder.EQUAL;
	if ( ! exact )
	    cmp_op = QueryBuilder.CASE_INSENSITIVE_CONTAINS;
	setQueryTitre( x, cmp_op );
    }

    /**
     * Set the titre to query
     * @param x The titre of the info to query.
     * @exception DataObjectException If a database access error occurs.
     */
    public void setQueryTitre( 
				String x )
    throws DataObjectException, QueryException {
	setQueryTitre( x, true );
    }

    /**
     * Add titre to the ORDER BY clause.
     *
     * @param direction_flag  True for ascending order, false for descending
     */
    public void addOrderByTitre(boolean direction_flag) {
	hitDb = true;
        builder.addOrderByColumn("titre",
					(direction_flag) ? "ASC" : "DESC");
    }


    /**
     * Add titre to the ORDER BY clause.  This convenience
     * method assumes ascending order.
     */
    public void addOrderByTitre() {
        builder.addOrderByColumn("titre","ASC");
    }


    /**
     * Set the contenu to query, with a QueryBuilder comparison operator.
     *
     * @param x The contenu of the info to query.
     * @param cmp_op QueryBuilder comparison operator to use.
     * @exception DataObjectException If a database access error occurs.
     * @exception QueryException If comparison operator is inappropriate
     * (e.g. CASE_SENSITIVE_CONTAINS on an integer field).
     */
    public void setQueryContenu(
				String x,
				String cmp_op )
    throws DataObjectException, QueryException
    {
	// Remove from cacheHits any DOs that do not meet this
	// setQuery requirement.
	for ( int i = 0; i < cacheHits.size() && ! hitDb; i++ ) {
	    InfoDO DO = ( InfoDO ) cacheHits.elementAt( i );
	    if ( null == DO ) continue;
	    boolean found = false;
	    String m = DO.getContenu();
	    found = QueryBuilder.compare( m, x, cmp_op );
	    if ( ! found )
		cacheHits.removeElementAt( i-- );
	}

	// Also prepare the SQL needed to query the database 
	// in case there is no cache, or the query involves other tables.
	if ( partialCache || hitDb )
	    builder.addWhere( InfoDO.Contenu, x, cmp_op );
    }

    /**
     * Set the contenu to query.
     *
     * @param x The contenu of the info to query.
     * @param exact to use matches or not
     * @exception DataObjectException If a database access error occurs.
     * @deprecated Instead use 
     * setQueryContenu(String x,String cmp_op)
     */
    public void setQueryContenu(
				String x, boolean exact)
    throws DataObjectException, QueryException
    {
	// TODO:  DELETE THIS METHOD IN A FUTURE RELEASE
	String cmp_op = QueryBuilder.EQUAL;
	if ( ! exact )
	    cmp_op = QueryBuilder.CASE_INSENSITIVE_CONTAINS;
	setQueryContenu( x, cmp_op );
    }

    /**
     * Set the contenu to query
     * @param x The contenu of the info to query.
     * @exception DataObjectException If a database access error occurs.
     */
    public void setQueryContenu( 
				String x )
    throws DataObjectException, QueryException {
	setQueryContenu( x, true );
    }

    /**
     * Add contenu to the ORDER BY clause.
     *
     * @param direction_flag  True for ascending order, false for descending
     */
    public void addOrderByContenu(boolean direction_flag) {
	hitDb = true;
        builder.addOrderByColumn("contenu",
					(direction_flag) ? "ASC" : "DESC");
    }


    /**
     * Add contenu to the ORDER BY clause.  This convenience
     * method assumes ascending order.
     */
    public void addOrderByContenu() {
        builder.addOrderByColumn("contenu","ASC");
    }

    /**
    * Returns the <code>QueryBuilder</code> that this <code>InfoQuery</code>
    * uses to construct and execute database queries.
    * <code>InfoQuery.setQueryXXX</code> methods use 
    * the <code>QueryBuilder</code> to
    * append SQL expressions to the <code>"WHERE"</code> clause to be executed.
    * The <code>QueryBuilder.addEndClause method.</code> can be used to
    * append freeform SQL expressions to the <code>WHERE</code> clause,
    * e.g. "ORDER BY name".
    *
    * <b>Notes regarding cache-enabled DO classes:</b>
    * DO classes can be cache-enabled.  
    * If when using a <code>InfoQuery</code>, the application developer
    * <b>does not call</b> <code>getQueryBuilder</code>,
    * then <code>InfoQuery.setQueryXXX</code> methods 
    * simply prune the DO cache and return the remaining results.
    * However, a <code>QueryBuilder</code> builds
    * <CODE>SELECT</CODE> statements for execution by the actual database,
    * and never searches the built-in cache for the DO.
    * So, if the DO class is cache-enabled, and <code>getQueryBuilder</code>
    * is called, this <CODE>InfoQuery</CODE> object ignores the cache 
    * and hits the actual database.
    */
    public QueryBuilder getQueryBuilder() {
	hitDatabase();
	return builder;
    }

    /**
     * Insert an <CODE>OR</CODE> between <CODE>WHERE</CODE> clauses.
     * Example:  find all the persons named Bob or Robert:
     * <CODE>
     *    PersonQuery pq = new PersonQuery();
     *    pq.setQueryFirstName( "Bob" );
     *    pq.or();
     *    pq.setQueryFirstName( "Robert" );
     * </CODE>
     * 
     * Note:  Calls to <CODE>setQueryXxx</CODE> methods
     * are implicitly <CODE>AND</CODE>ed together,
     * so the following example will always return nothing:
     * <CODE>
     *    PersonQuery pq = new PersonQuery();
     *    pq.setQueryFirstName( "Bob" );
     *    // AND automatically inserted here.
     *    pq.setQueryFirstName( "Robert" );
     * </CODE>
     * 
     * @see QueryBuilder to construct more elaborate queries.
     * @author Jay Gunter
     */
    public void or() {
	hitDb = true;
	builder.addWhereOr();
    }

    /**
     * Place an open parenthesis in the <CODE>WHERE</CODE> clause.
     * Example usage:  find all the Bobs and Roberts who are 5 or 50 years old:
     * <CODE>
     *    PersonQuery pq = new PersonQuery();
     *    pq.openParen();
     *       pq.setQueryFirstName( "Bob" );
     *       pq.or();
     *       pq.setQueryFirstName( "Robert" );
     *    pq.closeParen();
     *    // AND automatically inserted here.
     *    pq.openParen();
     *       pq.setQueryAge( 5 );
     *       pq.or();
     *       pq.setQueryAge( 50 );
     *    pq.closeParen();
     * </CODE>
     * 
     * @see QueryBuilder to construct more elaborate queries.
     * @author Jay Gunter
     */
    public void openParen() {
        hitDb = true;
	builder.addWhereOpenParen();
    }

    /**
     * Place a closing parenthesis in the <CODE>WHERE</CODE> clause.
     * 
     * @see openParen
     * @author Jay Gunter
     */
    public void closeParen() {
        hitDb = true;
	builder.addWhereCloseParen();
    }
}
